=begin

  Program:   Visualization Toolkit
  Module:    $RCSfile: Medical3.rb,v $
  Language:  Ruby
  Date:      $Date: 2005/03/11 07:50:08 $
  Version:   $Revision: 1.2 $

  Copyright (c) 1993-2002 Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=end


# This example reads a volume dataset, extracts two isosurfaces that
# represent the skin and bone, creates three orthogonal planes
# (saggital, axial, coronal), and displays them.
require "vtk"


DataPath = ENV["VTK_DATA_ROOT"] || raise("ERROR: Please define environment variable VTK_DATA_ROOT")
Prefix = DataPath + "/Data/headsq/quarter"



# Create the renderer, the render window, and the interactor. The
# renderer draws into the render window, the interactor enables
# mouse- and keyboard-based interaction with the data within the
# render window.
#
aRenderer = Vtk::Renderer.new
renWin = Vtk::RenderWindow.new
renWin.AddRenderer(aRenderer)
iren = Vtk::RenderWindowInteractor.new
iren.SetRenderWindow(renWin)

# The following reader is used to read a series of 2D slices (images)
# that compose the volume. The slice dimensions are set, and the
# pixel spacing. The data Endianness must also be specified. The
# reader usese the FilePrefix in combination with the slice number to
# construct filenames using the format FilePrefix.%d. (In this case
# the FilePrefix is the root name of the file: quarter.)
v16 = Vtk::Volume16Reader.new
v16.SetDataDimensions(64,64)
v16.SetDataByteOrderToLittleEndian
v16.SetFilePrefix(Prefix)
v16.SetImageRange(1, 93)
v16.SetDataSpacing(3.2, 3.2, 1.5)

# An isosurface, or contour value of 500 is known to correspond to
# the skin of the patient. Once generated, a vtkPolyDataNormals
# filter is is used to create normals for smooth surface shading
# during rendering.  The triangle stripper is used to create triangle
# strips from the isosurface; these render much faster on may
# systems.
skinExtractor = Vtk::ContourFilter.new
skinExtractor.SetInput( v16.GetOutput )
skinExtractor.SetValue(0, 500)
skinNormals = Vtk::PolyDataNormals.new
skinNormals.SetInput(skinExtractor.GetOutput)
skinNormals.SetFeatureAngle(60.0)
skinStripper = Vtk::Stripper.new
skinStripper.SetInput(skinNormals.GetOutput)
skinMapper = Vtk::PolyDataMapper.new
skinMapper.SetInput(skinStripper.GetOutput)
skinMapper.ScalarVisibilityOff
skin = Vtk::Actor.new
skin.SetMapper(skinMapper)
skin.GetProperty.SetDiffuseColor(1, 0.49, 0.25)
skin.GetProperty.SetSpecular(0.3)
skin.GetProperty.SetSpecularPower(20)

# An isosurface, or contour value of 1150 is known to correspond to
# the skin of the patient. Once generated, a vtkPolyDataNormals
# filter is is used to create normals for smooth surface shading
# during rendering.  The triangle stripper is used to create triangle
# strips from the isosurface; these render much faster on may
# systems.
boneExtractor = Vtk::ContourFilter.new
boneExtractor.SetInput(v16.GetOutput)
boneExtractor.SetValue(0, 1150)
boneNormals = Vtk::PolyDataNormals.new
boneNormals.SetInput(boneExtractor.GetOutput)
boneNormals.SetFeatureAngle(60.0)
boneStripper = Vtk::Stripper.new
boneStripper.SetInput(boneNormals.GetOutput)
boneMapper = Vtk::PolyDataMapper.new
boneMapper.SetInput(boneStripper.GetOutput)
boneMapper.ScalarVisibilityOff
bone = Vtk::Actor.new
bone.SetMapper(boneMapper)
bone.GetProperty.SetDiffuseColor(1, 1, 0.9412)

# An outline provides context around the data.
#
outlineData = Vtk::OutlineFilter.new
outlineData.SetInput(v16.GetOutput)
mapOutline = Vtk::PolyDataMapper.new
mapOutline.SetInput(outlineData.GetOutput)
outline = Vtk::Actor.new
outline.SetMapper(mapOutline)
outline.GetProperty.SetColor(0,0,0)

# Now we are creating three orthogonal planes passing through the
# volume. Each plane uses a different texture map and therefore has
# diferent coloration.

# Start by creatin a black/white lookup table.
bwLut = Vtk::LookupTable.new
bwLut.SetTableRange(0, 2000)
bwLut.SetSaturationRange(0, 0)
bwLut.SetHueRange(0, 0)
bwLut.SetValueRange(0, 1)

# Now create a lookup table that consists of the full hue circle
# (from HSV).
hueLut = Vtk::LookupTable.new
hueLut.SetTableRange(0, 2000)
hueLut.SetHueRange(0, 1)
hueLut.SetSaturationRange(1, 1)
hueLut.SetValueRange(1, 1)

# Finally, create a lookup table with a single hue but having a range
# in the saturation of the hue.
satLut = Vtk::LookupTable.new
satLut.SetTableRange(0, 2000)
satLut.SetHueRange(0.6, 0.6)
satLut.SetSaturationRange(0, 1)
satLut.SetValueRange(1, 1)

# Create the first of the three planes. The filter vtkImageMapToColors
# maps the data through the corresponding lookup table created above.  The
# vtkImageActor is a type of vtkProp and conveniently displays an image on
# a single quadrilateral plane. It does this using texture mapping and as
# a result is quite fast. (Note: the input image has to be unsigned char
# values, which the vtkImageMapToColors produces.) Note also that by
# specifying the DisplayExtent, the pipeline requests data of this extent
# and the vtkImageMapToColors only processes a slice of data.
saggitalColors = Vtk::ImageMapToColors.new
saggitalColors.SetInput(v16.GetOutput)
saggitalColors.SetLookupTable(bwLut)
saggital = Vtk::ImageActor.new
saggital.SetInput(saggitalColors.GetOutput)
saggital.SetDisplayExtent(32,32, 0,63, 0,92)

# Create the second (axial) plane of the three planes. We use the
# same approach as before except that the extent differs.
axialColors = Vtk::ImageMapToColors.new
axialColors.SetInput(v16.GetOutput)
axialColors.SetLookupTable(hueLut)
axial = Vtk::ImageActor.new
axial.SetInput(axialColors.GetOutput)
axial.SetDisplayExtent(0,63, 0,63, 46,46)

# Create the third (coronal) plane of the three planes. We use 
# the same approach as before except that the extent differs.
coronalColors = Vtk::ImageMapToColors.new
coronalColors.SetInput(v16.GetOutput)
coronalColors.SetLookupTable(satLut)
coronal = Vtk::ImageActor.new
coronal.SetInput(coronalColors.GetOutput)
coronal.SetDisplayExtent(0,63, 32,32, 0,92)

# It is convenient to create an initial view of the data. The
# FocalPoint and Position form a vector direction. Later on
# (ResetCamera method) this vector is used to position the camera
# to look at the data in this direction.
aCamera = Vtk::Camera.new
aCamera.SetViewUp(0, 0, -1)
aCamera.SetPosition(0, 1, 0)
aCamera.SetFocalPoint(0, 0, 0)
aCamera.ComputeViewPlaneNormal

# Actors are added to the renderer. 
aRenderer.AddActor(outline)
aRenderer.AddActor(saggital)
aRenderer.AddActor(axial)
aRenderer.AddActor(coronal)
aRenderer.AddActor(axial)
aRenderer.AddActor(coronal)
aRenderer.AddActor(skin)
aRenderer.AddActor(bone)

# Turn off bone for this example.
bone.VisibilityOff

# Set skin to semi-transparent.
skin.GetProperty.SetOpacity(0.5)

# An initial camera view is created.  The Dolly method moves 
# the camera towards the FocalPoint, thereby enlarging the image.
aRenderer.SetActiveCamera(aCamera)
aRenderer.ResetCamera 
aCamera.Dolly(1.5)

# Set a background color for the renderer and set the size of the
# render window (expressed in pixels).
aRenderer.SetBackground(1,1,1)
renWin.SetSize(640, 480)

# Note that when camera movement occurs (as it does in the Dolly
# method), the clipping planes often need adjusting. Clipping planes
# consist of two planes: near and far along the view direction. The 
# near plane clips out objects in front of the plane; the far plane
# clips out objects behind the plane. This way only what is drawn
# between the planes is actually rendered.
aRenderer.ResetCameraClippingRange 

# interact with data
iren.Initialize
iren.Start 
