/*=========================================================================

  Program:   Visualization Toolkit
  Module:    $RCSfile: vtkImageGridSource.h,v $
  Language:  C++
  Date:      $Date: 2005/03/11 02:32:16 $
  Version:   $Revision: 1.1 $

  Copyright (c) 1993-2002 Ken Martin, Will Schroeder, Bill Lorensen 
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even 
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
// .NAME vtkImageGridSource - Create an image of a grid.
// .SECTION Description
// vtkImageGridSource produces an image of a grid.  The
// default output type is float.

#ifndef __vtkImageGridSource_h
#define __vtkImageGridSource_h

#include "vtkImageSource.h"

class VTK_IMAGING_EXPORT vtkImageGridSource : public vtkImageSource
{
public:
  static vtkImageGridSource *New();
  vtkTypeRevisionMacro(vtkImageGridSource,vtkImageSource);
  void PrintSelf(ostream& os, vtkIndent indent);

  // Description:
  // Set/Get the grid spacing in pixel units.  Default (10,10,0).
  // A value of zero means no grid.
  vtkSetVector3Macro(GridSpacing,int);
  vtkGetVector3Macro(GridSpacing,int);
  
  // Description:
  // Set/Get the grid origin, in ijk integer values.  Default (0,0,0).
  vtkSetVector3Macro(GridOrigin,int);
  vtkGetVector3Macro(GridOrigin,int);

  // Description:
  // Set the grey level of the lines. Default 1.0.
  vtkSetMacro(LineValue,float);
  vtkGetMacro(LineValue,float);

  // Description:
  // Set the grey level of the fill. Default 0.0.
  vtkSetMacro(FillValue,float);
  vtkGetMacro(FillValue,float);

  // Description:
  // Set/Get the data type of pixels in the imported data.
  // As a convenience, the OutputScalarType is set to the same value.
  vtkSetMacro(DataScalarType,int);
  void SetDataScalarTypeToFloat(){this->SetDataScalarType(VTK_FLOAT);}
  void SetDataScalarTypeToInt(){this->SetDataScalarType(VTK_INT);}
  void SetDataScalarTypeToShort(){this->SetDataScalarType(VTK_SHORT);}
  void SetDataScalarTypeToUnsignedShort()
    {this->SetDataScalarType(VTK_UNSIGNED_SHORT);}
  void SetDataScalarTypeToUnsignedChar()
    {this->SetDataScalarType(VTK_UNSIGNED_CHAR);}
  vtkGetMacro(DataScalarType, int);
  const char *GetDataScalarTypeAsString() { 
    return vtkImageScalarTypeNameMacro(this->DataScalarType); }

  // Description:
  // Set/Get the extent of the whole output image,
  // Default: (0,255,0,255,0,0)
  vtkSetVector6Macro(DataExtent,int);
  vtkGetVector6Macro(DataExtent,int);

  // Description:
  // Set/Get the pixel spacing.
  vtkSetVector3Macro(DataSpacing,float);
  vtkGetVector3Macro(DataSpacing,float);
  
  // Description:
  // Set/Get the origin of the data.
  vtkSetVector3Macro(DataOrigin,float);
  vtkGetVector3Macro(DataOrigin,float);

protected:
  vtkImageGridSource();
  ~vtkImageGridSource() {};

  int GridSpacing[3];
  int GridOrigin[3];

  float LineValue;
  float FillValue;

  int DataScalarType;

  int DataExtent[6];
  float DataSpacing[3];
  float DataOrigin[3];

  virtual void ExecuteInformation();
  virtual void ExecuteData(vtkDataObject *data);
private:
  vtkImageGridSource(const vtkImageGridSource&);  // Not implemented.
  void operator=(const vtkImageGridSource&);  // Not implemented.
};


#endif
